import asyncio
import json
import os

from datetime import datetime

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal

from fleming.module_test.module_test_util import *


# >>>>>>>NOT TESTED YET<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

# Detector-Test Prototype 3-6 according to
# https://confluence-ext.perkinelmer.com/pages/viewpage.action?pageId=416913531
# Intention: Module Test of Detectors:
# Reference/Absorbance Photodiode pd 
# Photomultiplier pmt

#     Created: 2021-03-18/Kay Struebing
# Last Change: 2021-05-12/Kay Struebing
#      Tested: 2021-05-12/Kay Struebing
#     Version: 0.1
# 

# Global Constants
PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]


async def pdd_scan(iterations=1):
    '''
    ml 210527:
    [ ] 2021-05-31/kstruebing:
     - al_set_curent -> al_setpower
     - Change from Tab-separated to semicolon-separated
     - al_off added
     - reftest
    '''
    TEST_NAME = 'pdd_scan'
    #exptimes = [500, 1000, 2000, 5000, 10000, 20000, 50000, 100000]    # exposuretime in µs
    exp_1 = range(500, 200000, 1000)
    alphacurrent = 1024    # Alpha-Laser Current (0..4095)
    alphadelay = 2
    delay = 1
    # Set mover positions
    await mover_pdtest()
    # Switch on Alpha Laser
    await al_settemp()
    await al_setpower(alphacurrent)
    await al_enable()
    await al_on()
    await asyncio.sleep(alphadelay)
    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    start_time = datetime.now().strftime('%H:%M:%S')
    msg = f"Start {BASE_NAME}.{TEST_NAME} at {start_time}, {file_name}"
    PyLogger.logger.info(msg)
    await send_msg(json.dumps({'result': msg}))
    with open(file_name, 'a') as data:
        data.write(f"{file_name} started at {start_time}\n")
        data.write(f"alphacurrent = {alphacurrent}, alphadelay = {alphadelay}\n ")
        data.write("exposuretime;REF_Low;REF_High;ABS_Low;ABS_High\n")
        for exposuretime in exp_1:
            result = await reftest(iterations, exposuretime, delay)
            data.write(f"{exposuretime};{result[0]};{result[1]};{result[2]};{result[3]}\n")
    await al_off()
    return f"{TEST_NAME} Done, Output written to {file_name}"


async def mover_pdtest():
    # Todo:
    # - add adjusted positions
    fms = VUnits.instance.hal.filterModuleSlider
    els = VUnits.instance.hal.excitationLightSelector
    bld = VUnits.instance.hal.bottomLightDirector
    await fms.Home()    
    await fms.UseProfile(1)
    await fms.Move(35.8)
    await els.Home()    
    await els.UseProfile(1)
    await els.Move(245.8)
    await bld.Home()    
    await bld.UseProfile(1)
    await bld.Move(-6.94)


async def reftest(iterations=1, exposuretime=100, delay=1):
    '''
    ml 210527:
    test reference- and absorbance-diode
    ks 210531
    1 Time-Unit = 10ns
    [ ] 2021-06-09/kstruebing:
     - delay after range-switch
    '''
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    exptime = exposuretime * 100
    fixed_range = 500   # 5 µs
    resettime = 100000  # 1 ms

    op_id = 'reftest'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)
    seq_gen.TimerWaitAndRestart(resettime)
    seq_gen.currSequence.append(0x0F000000 | (2 << 15) | (2 << 12))     # SetIntegratorMode(IntModeAbs=full_reset, IntModeRef=full_reset)
    seq_gen.TimerWaitAndRestart(exptime - fixed_range)
    seq_gen.currSequence.append(0x0F000000 | (4 << 15) | (4 << 12))     # SetIntegratorMode(IntModeAbs=auto_range, IntModeRef=auto_range)
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.currSequence.append(0x0F000000 | (5 << 15) | (5 << 12))     # SetIntegratorMode(IntModeAbs=fixed_range, IntModeRef=fixed_range)
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(channel=4, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(channel=5, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.currSequence.append(0x0F000000 | (2 << 15) | (2 << 12))     # SetIntegratorMode(IntModeAbs=full_reset, IntModeRef=full_reset)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 4)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    resultsum = [0] * 4
    for i in range(iterations):
        try:
            await measurement_unit.ExecuteMeasurement(op_id)
            await asyncio.sleep(delay)
            results = await measurement_unit.ReadMeasurementValues(op_id)
            for index in range(0, len(results)):
                resultsum[index] = resultsum[index] + results[index]
        except BaseException as ex:
            msg = f"reftest() failed: {ex}"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({'result': msg}))

    return resultsum


async def pmt1_dscan(start=0, stop=4096, step=1, window=100):
    '''
    ml 210224:
    discrscan
    [x] 2021-04-27/kstruebing:
    hw.eef...-> eef = get_node_endpoint...
    [ ] 2021-06-09/kstruebing:
    data aquisition
    rename discrscan1 -> pmt1_dscan
    '''
    TEST_NAME = 'pmt1_dscan'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    pre_cnt_window = 100000    # 1ms
    delay = 0.1

    await hvpmt1off()
    await hvgate1on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"Discriminatorlevel Scan PMT 1, {time_stamp}\n")
        data.write(f"dac; count (window = {window} ms)\n")
        for i in range(start, stop, step):
            val = float(i) / 4096
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, val, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{i}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"Discr Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate1off()
    await hvpmt1on()
    return f"{TEST_NAME} done"


async def pmt2_dscan(start=0, stop=4096, step=1, window=100):
    '''
    ml 210224:
    discrscan
    [x] 2021-04-27/kstruebing:
    hw.eef...-> eef = get_node_endpoint...
    [ ] 2021-06-09/kstruebing:
    data aquisition
    rename discrscan2 -> pmt2_dscan
    '''
    TEST_NAME = 'pmt2_dscan'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    pre_cnt_window = 100000
    delay = 0.1

    await hvpmt2off()
    await hvgate2on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"Discriminatorlevel Scan PMT 2, {time_stamp}\n")
        data.write(f"dac; count (window = {window} ms)\n")
        for i in range(start, stop, step):
            val = float(i) / 4096
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, val, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{i}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"Discr Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate2off()
    await hvpmt2on()
    return f"{TEST_NAME} done"


async def pmt1_hscan(discr=1200, start=1000, stop=2500, step=1, window=100):
    '''
    ml 210224:
    hvscan
    [x] 2021-04-27/kstruebing:
    hw.eef...-> eef = get_node_endpoint...
    [ ] 2021-06-09/kstruebing:
    data aquisition
    rename hvscan1 -> pmt1_hscan
    discr = 1500 -> 1200
    start = 0 -> 1000
    stop = 2000 -> 2500
    '''
    TEST_NAME = 'pmt1_hscan'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    pre_cnt_window = 100000
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, (float(discr) / 4096), timeout=1)
    await hvpmt1on()
    await hvgate1on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"HV Scan PMT 1, {time_stamp}\n")
        data.write(f"dac; count (window = {window} ms)\n")
        for i in range(start, stop, step):
            val = float(i) / 4096
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, val, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{i}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"HV Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate1off()
    await hvpmt1off()
    return f"{TEST_NAME} done"


async def pmt2_hscan(discr=1200, start=1000, stop=2500, step=1, window=100):
    '''
    ml 210224:
    hvscan
    [x] 2021-04-27/kstruebing:
    hw.eef...-> eef = get_node_endpoint...
    [ ] 2021-06-09/kstruebing:
    rename hvscan2 -> pmt2_hscan
    data aquisition
    discr = 1500 -> 1200
    start = 0 -> 1000
    stop = 2000 -> 2500
    '''
    TEST_NAME = 'pmt2_hscan'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    pre_cnt_window = 100000
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, (float(discr) / 4096), timeout=1)
    await hvpmt2on()
    await hvgate2on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.Loop(window)
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
    seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"HV Scan PMT 2, {time_stamp}\n")
        data.write(f"dac; count (window = {window} ms)\n")
        for i in range(start, stop, step):
            val = float(i) / 4096
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, val, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{i}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"HV Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate2off()
    await hvpmt2off()
    return f"{TEST_NAME} done"


async def pmt1_dscan_v7(dl_start=0.0, dl_stop=1.0, dl_step=0.001, window_ms=100):
    """
    PMT1 Discriminator Level Scan for V7 hardware.
    """
    TEST_NAME = 'pmt1_dscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0, timeout=1)
    await hvgate1on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"Discriminator Level Scan PMT 1, {time_stamp}\n")
        data.write(f"dl; count (window = {window_ms} ms)\n")
        dl_range = [dl / 1e6 for dl in range(round(dl_start * 1e6), round(dl_stop * 1e6 + 1), round(dl_step * 1e6))]
        for dl in dl_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{dl:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"Discr Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate1off()
    return f"{TEST_NAME} done"


async def pmt2_dscan_v7(dl_start=0.0, dl_stop=1.0, dl_step=0.001, window_ms=100):
    """
    PMT2 Discriminator Level Scan for V7 hardware.
    """
    TEST_NAME = 'pmt2_dscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 0, timeout=1)
    await hvgate2on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"Discriminator Level Scan PMT 2, {time_stamp}\n")
        data.write(f"dl; count (window = {window_ms} ms)\n")
        dl_range = [dl / 1e6 for dl in range(round(dl_start * 1e6), round(dl_stop * 1e6 + 1), round(dl_step * 1e6))]
        for dl in dl_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{dl:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"Discr Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate2off()
    return f"{TEST_NAME} done"


async def pmt3_dscan_v7(dl_start=0.0, dl_stop=1.0, dl_step=0.001, window_ms=100):
    """
    PMT-US-LUM Discriminator Level Scan for V7 hardware.
    """
    TEST_NAME = 'pmt3_dscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, 0, timeout=1)

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=2, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"Discriminator Level Scan PMT-US-LUM, {time_stamp}\n")
        data.write(f"dl; count (window = {window_ms} ms)\n")
        dl_range = [dl / 1e6 for dl in range(round(dl_start * 1e6), round(dl_stop * 1e6 + 1), round(dl_step * 1e6))]
        for dl in dl_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, dl, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{dl:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"Discr Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))

    return f"{TEST_NAME} done"


async def pmt1_hscan_v7(dl=0.3, hv_start=0.2, hv_stop=0.6, hv_step=0.001, window_ms=100):
    """
    PMT1 High Voltage Setting Scan for V7 hardware.
    """
    TEST_NAME = 'pmt1_hscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl, timeout=1)
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 1, timeout=1)
    await hvgate1on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=0, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=0, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=0, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"HV Scan PMT 1, {time_stamp}\n")
        data.write(f"hv; count (window = {window_ms} ms)\n")
        hv_range = [hv / 1e6 for hv in range(round(hv_start * 1e6), round(hv_stop * 1e6 + 1), round(hv_step * 1e6))]
        for hv in hv_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{hv:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"HV Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate1off()
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, 0, timeout=1)
    return f"{TEST_NAME} done"


async def pmt2_hscan_v7(dl=0.3, hv_start=0.2, hv_stop=0.6, hv_step=0.001, window_ms=100):
    """
    PMT2 High Voltage Setting Scan for V7 hardware.
    """
    TEST_NAME = 'pmt2_hscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl, timeout=1)
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 1, timeout=1)
    await hvgate2on()

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=1, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"HV Scan PMT 2, {time_stamp}\n")
        data.write(f"hv; count (window = {window_ms} ms)\n")
        hv_range = [hv / 1e6 for hv in range(round(hv_start * 1e6), round(hv_stop * 1e6 + 1), round(hv_step * 1e6))]
        for hv in hv_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, hv, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{hv:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"HV Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))
    await hvgate2off()
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, 0, timeout=1)
    return f"{TEST_NAME} done"


async def pmt3_hscan_v7(dl=0.3, hv_start=0.2, hv_stop=0.6, hv_step=0.001, window_ms=100):
    """
    PMT-US-LUM High Voltage Setting Scan for V7 hardware.
    """
    TEST_NAME = 'pmt3_hscan_v7'
    measurement_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    window_us = round(window_ms * 1000)
    window_corse, window_fine = divmod(window_us, 65536)
    pre_cnt_window = 100    # 1 µs
    delay = 0.1

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, dl, timeout=1)
    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, 1, timeout=1)

    op_id = TEST_NAME
    seq_gen = meas_seq_generator()
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(channel=2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if (window_corse > 0):
        seq_gen.Loop(window_corse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if (window_fine > 0):
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(channel=2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=False)
        seq_gen.LoopEnd()
    seq_gen.GetPulseCounterResult(channel=2, relative=False, resetCounter=True, cumulative=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.Stop(0)
    measurement_unit.ClearOperations()
    measurement_unit.resultAddresses[op_id] = range(0, 1)
    await measurement_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    await send_msg(json.dumps({'result': f"Start {BASE_NAME}.{TEST_NAME}, {file_name}"}))
    with open(file_name, mode='a') as data:
        time_stamp = datetime.now().strftime('%Y-%m-%d_%H-%M')
        data.write(f"HV Scan PMT-US-LUM, {time_stamp}\n")
        data.write(f"hv; count (window = {window_ms} ms)\n")
        hv_range = [hv / 1e6 for hv in range(round(hv_start * 1e6), round(hv_stop * 1e6 + 1), round(hv_step * 1e6))]
        for hv in hv_range:
            await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, hv, timeout=1)
            await asyncio.sleep(delay)
            try:
                await measurement_unit.ExecuteMeasurement(op_id)
                results = await measurement_unit.ReadMeasurementValues(op_id)
                msg = f"{hv:.6f}; {results[0]}"
                data.write(msg + '\n')
                PyLogger.logger.info(msg)
                await send_msg(json.dumps({'result': msg}))
            except BaseException as ex:
                msg = f"HV Scan failed: {ex}"
                data.write(msg + '\n')
                PyLogger.logger.error(msg)
                await send_msg(json.dumps({'result': msg}))

    await measurement_unit.endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, 0, timeout=1)
    return f"{TEST_NAME} done"

